#!/bin/bash

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
NC='\033[0m'

CURRENT_DIR=$(
    cd "$(dirname "$0")" || exit
    pwd
)

function log() {
    message="[icontainer Log]: $1 "
    case "$1" in
        *"Falha"*|*"Erro"*|*"Por favor, execute este script com privilégios root ou sudo"*)
            echo -e "${RED}${message}${NC}" 2>&1 | tee -a "${CURRENT_DIR}"/install.log
            ;;
        *"Sucesso"*)
            echo -e "${GREEN}${message}${NC}" 2>&1 | tee -a "${CURRENT_DIR}"/install.log
            ;;
        *"Ignorar"*|*"Pular"*)
            echo -e "${YELLOW}${message}${NC}" 2>&1 | tee -a "${CURRENT_DIR}"/install.log
            ;;
        *)
            echo -e "${BLUE}${message}${NC}" 2>&1 | tee -a "${CURRENT_DIR}"/install.log
            ;;
    esac
}
echo
cat << 'EOF'
 _  ____            _        _                 
(_)/ ___|___  _ __ | |_ __ _(_)_ __   ___ _ __ 
| | |   / _ \| '_ \| __/ _` | | '_ \ / _ \ '__|
| | |__| (_) | | | | || (_| | | | | |  __/ |   
|_|\____\___/|_| |_|\__\__,_|_|_| |_|\___|_|   

EOF

log "======================= Iniciando a instalação ======================="

function Check_Root() {
    if [[ $EUID -ne 0 ]]; then
        log "Por favor, execute este script com privilégios root ou sudo"
        exit 1
    fi
}

function Prepare_System(){
    if which icontainer >/dev/null 2>&1; then
        log "O executável do painel icontainer já está instalado, por favor, não instale novamente"
        exit 1
    fi
}

function Set_Dir(){
    if read -t 120 -p "Defina o diretório de instalação do icontainer (padrão /etc): " PANEL_BASE_DIR; then
        if [[ "$PANEL_BASE_DIR" != "" ]]; then
            if [[ "$PANEL_BASE_DIR" != /* ]]; then
                log "Por favor, insira o caminho completo do diretório"
                Set_Dir
            fi

            if [[ ! -d $PANEL_BASE_DIR ]]; then
                mkdir -p "$PANEL_BASE_DIR"
                log "O caminho de instalação selecionado é $PANEL_BASE_DIR"
            fi
        else
            PANEL_BASE_DIR=/etc
            log "O caminho de instalação selecionado é $PANEL_BASE_DIR"
        fi
    else
        PANEL_BASE_DIR=/etc
        log "(Tempo de configuração esgotado, usando o caminho de instalação padrão /etc"
    fi
}

ACCELERATOR_URL="https://docker.io"
DAEMON_JSON="/etc/docker/daemon.json"
BACKUP_FILE="/etc/docker/daemon.json.icontainer_bak"

function create_daemon_json() {
    log "Criando novo arquivo de configuração ${DAEMON_JSON}..."
    mkdir -p /etc/docker
    echo '{
        "registry-mirrors": ["'"$ACCELERATOR_URL"'"]
    }' | tee "$DAEMON_JSON" > /dev/null
    log "Configuração de aceleração de imagem adicionada."
}

function configure_accelerator() {
    read -p "Deseja configurar a aceleração de imagem? (y/n): " configure_accelerator
    if [[ "$configure_accelerator" == "y" ]]; then
        if [ -f "$DAEMON_JSON" ]; then
            log "O arquivo de configuração já existe, faremos um backup do arquivo de configuração existente para ${BACKUP_FILE} e criaremos um novo arquivo de configuração."
            cp "$DAEMON_JSON" "$BACKUP_FILE"
            create_daemon_json
        else
            create_daemon_json
        fi

        log "Reiniciando o serviço Docker..."
        systemctl daemon-reload
        systemctl restart docker
        log "Serviço Docker reiniciado com sucesso."
    else
        log "A aceleração de imagem não foi configurada."
    fi
}

function Install_Docker(){
    if which docker >/dev/null 2>&1; then
        log "Docker detectado como instalado, pulando etapa de instalação"
        configure_accelerator
    else
        log "... Iniciando a instalação do docker"
	log "... Instalando dependências do docker"
	dnf install -y yum-utils device-mapper-persistent-data lvm2

	log "... Configurando repositório do docker"
	yum-config-manager --add-repo https://download.docker.com/linux/centos/docker-ce.repo

	 log "... Instalando docker"
	dnf install -y docker-ce docker-ce-cli containerd.io

    fi       
}

function Enable_Docker(){
	systemctl start docker
	systemctl enable docker
}


function Install_Compose(){
    docker-compose version >/dev/null 2>&1
    if [[ $? -ne 0 ]]; then
        log "... Instalando docker-compose online"
        
        arch=$(uname -m)
        if [ "$arch" == 'armv7l' ]; then
            arch='armv7'
        fi

	VERSION=$(curl -s https://api.github.com/repos/docker/compose/releases/latest | grep tag_name | cut -d '"' -f 4)
	curl -SL https://github.com/docker/compose/releases/download/${VERSION}/docker-compose-linux-x86_64 -o /usr/local/bin/docker-compose


        if [[ ! -f /usr/local/bin/docker-compose ]]; then
            log "Falha no download do docker-compose, por favor, tente novamente mais tarde"
            exit 1
        fi
        chmod +x /usr/local/bin/docker-compose
        ln -s /usr/local/bin/docker-compose /usr/bin/docker-compose

        docker-compose version >/dev/null 2>&1
        if [[ $? -ne 0 ]]; then
            log "Falha na instalação do docker-compose"
            exit 1
        else
            log "docker-compose instalado com sucesso"
        fi
    else
        compose_v=$(docker-compose -v)
        if [[ $compose_v =~ 'docker-compose' ]]; then
            read -p "Detectado que a versão do Docker Compose instalada é antiga (deve ser >= v2.0.0), deseja atualizar [y/n] : " UPGRADE_DOCKER_COMPOSE
            if [[ "$UPGRADE_DOCKER_COMPOSE" == "Y" ]] || [[ "$UPGRADE_DOCKER_COMPOSE" == "y" ]]; then
                rm -rf /usr/local/bin/docker-compose /usr/bin/docker-compose
                Install_Compose
            else
                log "A versão do Docker Compose é $compose_v, isso pode afetar o uso normal da loja de aplicativos"
            fi
        else
            log "Docker Compose detectado como instalado, pulando etapa de instalação"
        fi
    fi
}

function Set_Port(){
    DEFAULT_PORT=$(expr $RANDOM % 55535 + 10000)

    while true; do
        read -p "Defina a porta do iContainer (padrão $DEFAULT_PORT): " PANEL_PORT

        if [[ "$PANEL_PORT" == "" ]]; then
            PANEL_PORT=$DEFAULT_PORT
        fi

        if ! [[ "$PANEL_PORT" =~ ^[1-9][0-9]{0,4}$ && "$PANEL_PORT" -le 65535 ]]; then
            log "Erro: o número da porta deve estar entre 1 e 65535"
            continue
        fi

        if command -v ss >/dev/null 2>&1; then
            if ss -tlun | grep -q ":$PANEL_PORT " >/dev/null 2>&1; then
                log "A porta $PANEL_PORT está em uso, por favor, insira novamente..."
                continue
            fi
        elif command -v netstat >/dev/null 2>&1; then
            if netstat -tlun | grep -q ":$PANEL_PORT " >/dev/null 2>&1; then
                log "A porta $PANEL_PORT está em uso, por favor, insira novamente..."
                continue
            fi
        fi

        log "A porta definida é: $PANEL_PORT"
        break
    done
}

function Install_Firewall(){
	dnf install -y firewalld
	#systemctl start firewalld
	#systemctl enable firewalld
}


function Set_Firewall(){
    if which firewall-cmd >/dev/null 2>&1; then
        if systemctl status firewalld | grep -q "Active: active" >/dev/null 2>&1; then
            log "Abrindo a porta $PANEL_PORT no firewall"
            firewall-cmd --zone=public --add-port="$PANEL_PORT"/tcp --permanent
            firewall-cmd --zone=public --add-port=80/tcp --permanent
            firewall-cmd --zone=public --add-port=443/tcp --permanent
            firewall-cmd --zone=public --add-port=21/tcp --permanent
            firewall-cmd --reload
        else
            log "Firewall não está ativo, ignorando abertura de porta"
        fi
    fi

    if which ufw >/dev/null 2>&1; then
        if systemctl status ufw | grep -q "Active: active" >/dev/null 2>&1; then
            log "Abrindo a porta $PANEL_PORT no firewall"
            ufw allow "$PANEL_PORT"/tcp
            ufw reload
        else
            log "Firewall não está ativo, ignorando abertura de porta"
        fi
    fi
}

function Enable_Firewall(){
        systemctl start firewalld
        systemctl enable firewalld
}

function Set_Entrance(){
    DEFAULT_ENTRANCE=`cat /dev/urandom | head -n 16 | md5sum | head -c 10`

    while true; do
        read -p "Defina a entrada segura do iContainer (padrão $DEFAULT_ENTRANCE): " PANEL_ENTRANCE
        if [[ "$PANEL_ENTRANCE" == "" ]]; then
            PANEL_ENTRANCE=$DEFAULT_ENTRANCE
        fi

        if [[ ! "$PANEL_ENTRANCE" =~ ^[a-zA-Z0-9_]{3,30}$ ]]; then
            log "Erro: a entrada segura do painel suporta apenas letras, números e underscores, com 3-30 caracteres de comprimento"
            continue
        fi
    
        log "A entrada segura do painel definida é: $PANEL_ENTRANCE"
        break
    done
}

function Set_Username(){
    DEFAULT_USERNAME=$(cat /dev/urandom | head -n 16 | md5sum | head -c 10)

    while true; do
        read -p "Defina o usuário do painel iContainer (padrão $DEFAULT_USERNAME): " PANEL_USERNAME

        if [[ "$PANEL_USERNAME" == "" ]]; then
            PANEL_USERNAME=$DEFAULT_USERNAME
        fi

        if [[ ! "$PANEL_USERNAME" =~ ^[a-zA-Z0-9_]{5,30}$ ]]; then
            log "Erro: o usuário do painel suporta apenas letras, números e underscores, com 5-30 caracteres de comprimento"
            continue
        fi

        log "O usuário do painel definido é: $PANEL_USERNAME"
        break
    done
}


function passwd() {
    charcount='0'
    reply=''
    while :; do
        char=$(
            stty cbreak -echo
            dd if=/dev/tty bs=1 count=1 2>/dev/null
            stty -cbreak echo
        )
        case $char in
        "$(printenv '\000')")
            break
            ;;
        "$(printf '\177')" | "$(printf '\b')")
            if [ $charcount -gt 0 ]; then
                printf '\b \b'
                reply="${reply%?}"
                charcount=$((charcount - 1))
            else
                printf ''
            fi
            ;;
        "$(printf '\033')") ;;
        *)
            printf '*'
            reply="${reply}${char}"
            charcount=$((charcount + 1))
            ;;
        esac
    done
    printf '\n' >&2
}

function Set_Password(){
    DEFAULT_PASSWORD=$(cat /dev/urandom | head -n 16 | md5sum | head -c 10)

    while true; do
        log "Defina a senha do painel iContainer, após configurar, pressione enter para continuar (padrão $DEFAULT_PASSWORD):"
        passwd
        PANEL_PASSWORD=$reply
        if [[ "$PANEL_PASSWORD" == "" ]]; then
            PANEL_PASSWORD=$DEFAULT_PASSWORD
        fi

        if [[ ! "$PANEL_PASSWORD" =~ ^[a-zA-Z0-9_!@#$%*,.?]{8,30}$ ]]; then
            log "Erro: a senha do painel suporta apenas letras, números e caracteres especiais (!@#$%*_,.?), com 8-30 caracteres de comprimento"
            continue
        fi

        break
    done
}

function Init_Panel(){
    log "Configurando o Serviço iContainer"

    RUN_BASE_DIR=$PANEL_BASE_DIR/icontainer
    mkdir -p "$RUN_BASE_DIR"
    rm -rf "$RUN_BASE_DIR:?/*"

    cd "${CURRENT_DIR}" || exit

    cp ./icontainer /usr/bin && chmod +x /usr/bin/icontainer
    if [[ ! -f /usr/bin/icontainer ]]; then
        ln -s /usr/local/bin/icontainer /usr/bin/icontainer >/dev/null 2>&1
    fi

    cp ./icp /usr/local/bin && chmod +x /usr/local/bin/icp
    sed -i -e "s#BASE_DIR=.*#BASE_DIR=${PANEL_BASE_DIR}#g" /usr/local/bin/icp
    sed -i -e "s#ORIGINAL_PORT=.*#ORIGINAL_PORT=${PANEL_PORT}#g" /usr/local/bin/icp
    sed -i -e "s#ORIGINAL_USERNAME=.*#ORIGINAL_USERNAME=${PANEL_USERNAME}#g" /usr/local/bin/icp
    ESCAPED_PANEL_PASSWORD=$(echo "$PANEL_PASSWORD" | sed 's/[!@#$%*_,.?]/\\&/g')
    sed -i -e "s#ORIGINAL_PASSWORD=.*#ORIGINAL_PASSWORD=${ESCAPED_PANEL_PASSWORD}#g" /usr/local/bin/icp
    sed -i -e "s#ORIGINAL_ENTRANCE=.*#ORIGINAL_ENTRANCE=${PANEL_ENTRANCE}#g" /usr/local/bin/icp
    if [[ ! -f /usr/bin/icp ]]; then
        ln -s /usr/local/bin/icp /usr/bin/icp >/dev/null 2>&1
    fi

    cp ./icontainer.service /etc/systemd/system

    systemctl enable icontainer; systemctl daemon-reload 2>&1 | tee -a "${CURRENT_DIR}"/install.log

    log "Iniciando o serviço iContainer"
    systemctl start icontainer | tee -a "${CURRENT_DIR}"/install.log

    for b in {1..30}
    do
        sleep 3
        service_status=$(systemctl status icontainer 2>&1 | grep Active)
        if [[ $service_status == *running* ]]; then
            log "Serviço iContainer iniciado com sucesso!"
            break;
        else
            log "Erro ao iniciar o serviço iContainer!"
            exit 1
        fi
    done
    sed -i -e "s#ORIGINAL_PASSWORD=.*#ORIGINAL_PASSWORD=\*\*\*\*\*\*\*\*\*\*#g" /usr/local/bin/icp
}

function Get_Ip(){
    active_interface=$(ip route get 8.8.8.8 | awk 'NR==1 {print $5}')
    if [[ -z $active_interface ]]; then
        LOCAL_IP="127.0.0.1"
    else
        LOCAL_IP=$(ip -4 addr show dev "$active_interface" | grep -oP '(?<=inet\s)\d+(\.\d+){3}')
    fi

    PUBLIC_IP=$(curl -s https://api64.ipify.org)
    if [[ -z "$PUBLIC_IP" ]]; then
        PUBLIC_IP="N/A"
    fi
    if echo "$PUBLIC_IP" | grep -q ":"; then
        PUBLIC_IP=[${PUBLIC_IP}]
        icp listen-ip ipv6
    fi
}

function Show_Result(){
    log ""
    log "=================Obrigado por sua paciência, a instalação foi concluída=================="
    log ""
    log "Por favor, acesse o painel com o navegador:"
    log "Endereço externo: http://$PUBLIC_IP:$PANEL_PORT/$PANEL_ENTRANCE"
    log "Endereço interno: http://$LOCAL_IP:$PANEL_PORT/$PANEL_ENTRANCE"
    log "Usuário do painel: $PANEL_USERNAME"
    log "Senha do painel: $PANEL_PASSWORD"
    log ""
    log "Site oficial do projeto: https://icontainer.run"
    log "Documentação do projeto: https://icontainer/docs"
    log ""
    log "Se estiver usando um servidor em nuvem, por favor, abra a porta $PANEL_PORT no grupo de segurança"
    log ""
    log "Para a segurança do seu servidor, após sair desta tela você não poderá ver sua senha novamente, por favor, memorize sua senha."
    log ""
    log "================================================================"
}

function main(){
    Check_Root
    Prepare_System
    Set_Dir
    Install_Docker
    Enable_Docker
    Install_Compose
    Set_Port
    Install_Firewall
    Enable_Firewall
    Set_Firewall
    Set_Entrance
    Set_Username
    Set_Password
    Init_Panel
    Get_Ip
    Show_Result
}
main
